(() => {
    "use strict";

    // Only for google.com search pages (extra safety)
    if (location.hostname !== "www.google.com" || location.pathname !== "/search") return;

    const counts = new Map();

    function getDomainFromQuery() {
        const qInput = document.querySelector('input[name="q"]');
        if (!qInput) return null;
        const match = qInput.value.match(/site:([\w.-]+)/i);
        return match ? match[1].replace(/^www\./, "") : null;
    }

    function parseCount(text) {
        const t = (text || "").trim();
        const match =
            t.match(/(\d{1,3}(?:[,\s.]\d{3})+)/) || // 1,234 / 1 234 / 1.234
            t.match(/(\d+)/);
        return match && match[1] ? match[1] : null;
    }

    function getTotalCount() {
        // Most reliable on google.com
        const el = document.querySelector("#result-stats");
        if (el) {
            const n = parseCount(el.textContent || el.innerText);
            if (n) return n;
        }

        // Fallbacks (google can A/B test UI)
        const fallbacks = [".LHJvCe", 'div[role="heading"]'];
        for (const sel of fallbacks) {
            const e = document.querySelector(sel);
            if (!e) continue;
            const n = parseCount(e.textContent || e.innerText);
            if (n) return n;
        }

        return "0";
    }

    function ensureBadge() {
        let div = document.getElementById("custom-results-count");
        if (!div) {
            div = document.createElement("div");
            div.id = "custom-results-count";
            div.style.display = "inline-block";
            div.style.fontSize = "16px";
            div.style.border = "1px solid rgb(60,64,67)";
            div.style.borderRadius = "20px";
            div.style.padding = "10px 15px";
            div.style.marginBottom = "30px";
            div.textContent = "about … results"; // show instantly

            const container = document.querySelector("#search") || document.body;
            container.prepend(div);
        }
        return div;
    }

    function render() {
        if (location.pathname !== "/search") return;

        const domain = getDomainFromQuery();
        if (!domain) return;

        const badge = ensureBadge();

        const count = counts.get(domain) || getTotalCount();
        counts.set(domain, count);

        badge.textContent = `about ${count} results`;
    }

    // ---- FAST FIRST PAINT ----
    // 1) Render immediately (shows "…")
    ensureBadge();
    render();

    // 2) Fast poll for the first ~2 seconds to catch #result-stats ASAP
    const start = Date.now();
    const poll = setInterval(() => {
        render();
        if (Date.now() - start > 2000) clearInterval(poll);
    }, 120);

    // ---- LIGHT OBSERVER ----
    // Observe only relevant container (not whole DOM)
    function attachObserver() {
        const target = document.querySelector("#result-stats") || document.querySelector("#search");
        if (!target) {
            setTimeout(attachObserver, 120);
            return;
        }

        let scheduled = false;
        const schedule = () => {
            if (scheduled) return;
            scheduled = true;
            requestAnimationFrame(() => {
                scheduled = false;
                render();
            });
        };

        const obs = new MutationObserver(schedule);
        obs.observe(target, { childList: true, subtree: true, characterData: true });

        // Detect SPA URL changes
        let lastHref = location.href;
        setInterval(() => {
            if (location.href !== lastHref) {
                lastHref = location.href;
                // immediate update on navigation
                ensureBadge();
                render();
            }
        }, 400);
    }

    attachObserver();
})();
